﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using KeyConfig.Config;
using KeyConfig.Map;

namespace KeyConfig
{
	public class PadEditor : EditorBase
	{
		//----------// //----------// //----------// //----------//
		//
		//	Fields
		//
		//----------// //----------// //----------// //----------//

		// Controls
		Timer tmrMain;
		List<TextBox> textBoxList = new List<TextBox>();

		// Pad Device
		PadMonitor _padMonitor;
		int padValue = 0;

		// Config
		ShootingConfig _shootConfig;
		KeyNameMap _nameMap;

		//----------// //----------// //----------// //----------//
		//
		//	Constructor
		//
		//----------// //----------// //----------// //----------//

		/// <summary>
		/// PadEditor のインスタンスを作成します。
		/// </summary>
		public PadEditor()
		{
			tmrMain = new Timer()
			{
				Interval = 20
			};
			tmrMain.Tick += new EventHandler(tmrMain_Tick);
			tmrMain.Start();
		}

		//----------// //----------// //----------// //----------//
		//
		//	Methods
		//
		//----------// //----------// //----------// //----------//

		public void InitializeEditor(ShootingConfig config, KeyNameMap nameMap, GamePad gamePad, int labelWidth, int textWidth, int labelHeight)
		{
			if (config == null) throw new ArgumentNullException("config");
			if (nameMap == null) throw new ArgumentNullException("nameMap");

			_shootConfig = config;
			_nameMap = nameMap;

			if (gamePad != null)
			{
				_padMonitor = new PadMonitor(gamePad);
				_padMonitor.Pressed += new EventHandler<DataEventArgs<int>>(PadMonitor_Pressed);
			}

			var lblTitle = new Label()
			{
				AutoSize = true,
				Text = "パッド用コンフィグ"
			};

			this.Controls.Add(lblTitle);

			var kt = new ListLabel() { Top = 24 };
			kt.UpdateTitle(GetTitles(), 88);

			this.Controls.Add(kt);

			for (int i = 0; i < config.Pads.Count; i++)
			{
				var textBox = new TextBox()
				{
					Left = labelWidth + 8,
					Width = textWidth,
					Top = labelHeight * (i + 1),
					Text = config.Pads[i].ToString()
				};

				textBox.KeyDown += new KeyEventHandler(textBox_KeyDown);
				textBox.TextChanged += new EventHandler(textBox_TextChanged);

				this.Controls.Add(textBox);

				textBoxList.Add(textBox);
			}

			this.Width = textBoxList[textBoxList.Count - 1].Right;
			this.Height = labelHeight * (config.Pads.Count + 1);
		}

		private IEnumerable<string> GetTitles()
		{
			for (int i = 0; i < _shootConfig.Pads.Count; i++)
			{
				yield return _nameMap.Parse(i);
			}
		}

		/// <summary>
		/// パッドの入力値を反映する
		/// </summary>
		void SetPadValue()
		{
			for (int i = 0; i < textBoxList.Count; i++)
			{
				var box = textBoxList[i];
				if (box.Focused)
				{
					_shootConfig.Pads[i] = padValue;
					box.Text = padValue.ToString();

					if (i < textBoxList.Count - 1)
					{
						textBoxList[i + 1].Focus();
					}
					else
					{
						OnProceedFocus(EventArgs.Empty);
					}
					Edited = true;
					break;
				}
			}
		}

		//----------// //----------// //----------// //----------//
		//
		//	Event Handlers
		//
		//----------// //----------// //----------// //----------//

		void textBox_KeyDown(object sender, KeyEventArgs e)
		{
			if ((Keys.A <= e.KeyCode) && (e.KeyCode <= Keys.Z))
			{
				e.SuppressKeyPress = true;
			}
			else if (Keys.Oem1 <= e.KeyCode)
			{
				e.SuppressKeyPress = true;
			}
		}

		/// <summary>
		/// テキストボックスからの入力を受け付ける
		/// </summary>
		void textBox_TextChanged(object sender, EventArgs e)
		{
			var box = sender as TextBox;
			var index = textBoxList.IndexOf(box);

			try
			{
				var key = Convert.ToInt32(box.Text);
				_shootConfig.Pads[index] = key;
				Edited = true;
			}
			catch
			{
				box.Text = _shootConfig.Pads[index].ToString();
			}
		}

		/// <summary>
		/// ゲームパッドをポーリングする
		/// </summary>
		private void tmrMain_Tick(object sender, EventArgs e)
		{
			if (_padMonitor != null)
			{
				_padMonitor.Check();
			}
		}

		/// <summary>
		/// ゲームパッドからの入力を受け付ける
		/// </summary>
		void PadMonitor_Pressed(object sender, DataEventArgs<int> e)
		{
			if (this.IsDisposed) return;

			padValue = e.Value;
			this.Invoke(new System.Threading.ThreadStart(SetPadValue));
		}

		//----------// //----------// //----------// //----------//
		//
		//	Properties
		//
		//----------// //----------// //----------// //----------//

		internal Control FocusTarget
		{
			get { return textBoxList[0]; }
		}
	}
}