﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Text;

namespace KeyConfig.Config
{
	/// <summary>
	/// ini ファイルを操作するクラスです。
	/// </summary>
	public static class IniEditor
	{
		[DllImport("kernel32.dll")]
		private static extern uint GetPrivateProfileString(string lpAppName, string lpKeyName, string lpDefault,
			StringBuilder lpReturnedString, uint nSize, string lpFileName);

		[DllImport("kernel32.dll")]
		private static extern bool WritePrivateProfileString(string lpAppName, string lpKeyName, string lpString, string lpFileName);

		/// <summary>
		/// ini ファイルから文字列値を読み取ります。
		/// </summary>
		public static string Read(string lpAppName, string lpKeyName, string lpDefault, string lpFileName)
		{
			var sb = new StringBuilder(1024);
			GetPrivateProfileString(lpAppName, lpKeyName, lpDefault, sb, (uint)sb.Capacity, lpFileName);
			return sb.ToString();
		}

		/// <summary>
		/// ini ファイルから真偽値を読み取ります。
		/// </summary>
		public static bool ReadBoolean(string lpAppName, string lpKeyName, bool lpDefault, string lpFileName)
		{
			var text = Read(lpAppName, lpKeyName, lpDefault ? "1" : "0", lpFileName);

			return text == "1";
		}

		/// <summary>
		/// ini ファイルから整数値を読み取ります。
		/// </summary>
		public static int ReadInteger(string lpAppName, string lpKeyName, int lpDefault, string lpFileName)
		{
			try
			{
				var text = Read(lpAppName, lpKeyName, lpDefault.ToString(), lpFileName);
				return Convert.ToInt32(text);
			}
			catch
			{
				return lpDefault;
			}
		}

		/// <summary>
		/// ini ファイルに文字列値を書き込みます。
		/// </summary>
		public static void Write(string lpAppName, string lpKeyName, string lpString, string lpFileName)
		{
			WritePrivateProfileString(lpAppName, lpKeyName, lpString, lpFileName);
		}

		/// <summary>
		/// ini ファイルに真偽値を書き込みます。
		/// </summary>
		public static void WriteBoolean(string lpAppName, string lpKeyName, bool value, string lpFileName)
		{
			Write(lpAppName, lpKeyName, value ? "1" : "0", lpFileName);
		}

		/// <summary>
		/// ini ファイルに整数値を書き込みます。
		/// </summary>
		public static void WriteInteger(string lpAppName, string lpKeyName, int value, string lpFileName)
		{
			Write(lpAppName, lpKeyName, value.ToString(), lpFileName);
		}
	}
}