﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows.Forms;
using KeyConfig.Config;
using KeyConfig.Map;

namespace KeyConfig
{
	public partial class frmMain : Form
	{
		#region # Fields #

		// Config
		ShootingConfig _shootConfig;
		KeyNameMap _nameMap = null;

		// Controls
		KeyEditor keyEditor;
		PadEditor padEditor;
		ScreenEditor screenEditor;

		// DirectX
		GamePad _gamePad;


		// Window
		frmPad padWindow = null;

		// Flags
		bool initFailed = false;

		#endregion

		//----------// //----------// //----------// //----------// //----------// //----------//

		#region # Constructor, Destructor #
		/// <summary>
		/// frmMain のインスタンスを作成します。
		/// </summary>
		public frmMain()
		{
			InitializeComponent();
			Application.ThreadException += new ThreadExceptionEventHandler(Application_ThreadException);
		}

		private void frmMain_Load(object sender, EventArgs e)
		{
			DetectGamePad();
			ReadShootingConfig();
			InitializeEditor();

			this.ActiveControl = keyEditor.FocusTarget;
		}

		private void frmMain_FormClosing(object sender, FormClosingEventArgs e)
		{
			if (!initFailed)
			{
				if (Edited)
				{
					DialogResult dr = MessageBox.Show("保存しないで終了しますか？", "確認", MessageBoxButtons.OKCancel);

					if (dr != DialogResult.OK)
					{
						e.Cancel = true;
					}
				}
			}
			if (!e.Cancel)
			{
				if (_gamePad != null)
				{
					_gamePad.Dispose();
				}
			}
		}
		#endregion

		//----------// //----------// //----------// //----------// //----------// //----------//

		private void DetectGamePad()
		{
			try
			{
				_gamePad = new GamePad(this.Handle);
			}
			catch (Exception ex)
			{
				MessageBox.Show("ゲームパッドの検出中にエラーが発生しました。", Program.Title, MessageBoxButtons.OK, MessageBoxIcon.Warning);
				Program.WriteLog(ex.ToString());
			}
		}

		private void ReadShootingConfig()
		{
			try
			{
				_shootConfig = ShootingConfigReader.ReadFrom(KeyConfigPaths.ShootingIni);
			}
			catch (Exception ex)
			{
				_shootConfig = new ShootingConfig();

				MessageBox.Show("CONFIG.INI の読み取り中にエラーが発生しました。", Program.Title, MessageBoxButtons.OK, MessageBoxIcon.Warning);
				Program.WriteLog(ex.ToString());
			}
		}

		private void InitializeEditor()
		{
			try
			{
				_nameMap = KeyNameMapReader.ReadFrom(KeyConfigPaths.Dictionary);
				CreateComponents();
			}
			catch (Exception ex)
			{
				Program.WriteLog(ex.ToString());

				initFailed = true;
				Application.Exit();
			}
		}

		const int keyLeft = 8;
		const int padLeft = 200;

		const int labelTop = 32;

		const int labelWidth = 88;
		const int labelHeight = 24;
		const int textWidth = 64;

		const int keyTextLeft = keyLeft + labelWidth;
		const int padTextLeft = padLeft + labelWidth;

		private void CreateComponents()
		{
			CreateKeyEditor();
			CreatePadEditor();
			CreateScreenEditor();

			this.ClientSize = new Size(screenEditor.Right + 8, keyEditor.Bottom + 8);

			btnOK.Left = this.ClientSize.Width - btnOK.Width - 8;
			btnOK.Top = this.ClientSize.Height - btnOK.Height - 8;

			btnCancel.Left = btnOK.Left - btnCancel.Width - 8;
			btnCancel.Top = btnOK.Top;

			btnCheck.Left = btnCancel.Left - btnCheck.Width - 16;
			btnCheck.Top = btnOK.Top;
		}

		private void CreateKeyEditor()
		{
			keyEditor = new KeyEditor()
			{
				Left = 8,
				Top = 8
			};
			keyEditor.InitializeEditor(_shootConfig, _nameMap, labelWidth, textWidth, labelHeight);
			keyEditor.EditedChanged += new EventHandler(EditorBase_EditedChanged);
			keyEditor.ProceedFocus += new EventHandler(keyEditor_ProceedFocus);
			this.Controls.Add(keyEditor);
		}

		private void CreatePadEditor()
		{
			padEditor = new PadEditor()
			{
				Left = 200,
				Top = 8
			};
			padEditor.InitializeEditor(_shootConfig, _nameMap, _gamePad, labelWidth, textWidth, labelHeight);
			padEditor.EditedChanged += new EventHandler(EditorBase_EditedChanged);
			padEditor.ProceedFocus += new EventHandler(padEditor_ProceedFocus);
			this.Controls.Add(padEditor);
		}

		private void CreateScreenEditor()
		{
			screenEditor = new ScreenEditor()
			{
				Left = 392,
				Top = 8
			};
			screenEditor.InitializeEditor(_shootConfig);
			screenEditor.EditedChanged += new EventHandler(EditorBase_EditedChanged);

			this.Controls.Add(screenEditor);
		}

		//----------// //----------// //----------// //----------//
		//
		//	Event Handlers
		//
		//----------// //----------// //----------// //----------//

		void EditorBase_EditedChanged(object sender, EventArgs e)
		{
			if (Edited)
			{
				this.Text = String.Format("{0} *", Program.Title);
			}
			else
			{
				this.Text = Program.Title;
			}
		}

		void keyEditor_ProceedFocus(object sender, EventArgs e)
		{
			padEditor.FocusTarget.Focus();
		}

		void padEditor_ProceedFocus(object sender, EventArgs e)
		{
			screenEditor.FocusTarget.Focus();
		}

		private void btnOK_Click(object sender, EventArgs e)
		{
			try
			{
				ShootingConfigWriter.WriteTo(_shootConfig, KeyConfigPaths.ShootingIni);

				Edited = false;
				MessageBox.Show("保存しました。");
				this.Close();
			}
			catch (Exception ex)
			{
				MessageBox.Show("CONFIG.INI の書き込み中にエラーが発生しました。", Program.Title, MessageBoxButtons.OK, MessageBoxIcon.Warning);
				Program.WriteLog(ex.ToString());
			}
		}

		private void btnCancel_Click(object sender, EventArgs e)
		{
			this.Close();
		}

		private void btnCheck_Click(object sender, EventArgs e)
		{
			if ((padWindow == null) || (padWindow.IsDisposed))
			{
				padWindow = new frmPad();
				padWindow.InitializeEditor(_gamePad);
			}
			padWindow.Show();
			padWindow.BringToFront();
		}

		void Application_ThreadException(object sender, ThreadExceptionEventArgs e)
		{
			MessageBox.Show(e.Exception.ToString());
			initFailed = true;
			Application.Exit();
		}

		internal bool Edited
		{
			get
			{
				return keyEditor.Edited || padEditor.Edited || screenEditor.Edited;
			}
			set
			{
				keyEditor.Edited = padEditor.Edited = screenEditor.Edited = value;
			}
		}
	}
}